*************************************************************************
* sony_rcv.asm:  decodes the Sony format IR protocol                    *
*                                                                       *
* by Brian Silverman                            (bss@media.mit.edu)     *
* modified, ported to Handy Board by Fred Martin (fredm@media.mit.edu)  *
* August 15, 1997                                                       *
*************************************************************************

TIC1    EQU     $1010   ; Timer Input Capture register 1
TFLG1   EQU     $1023   ; main Timer interrupt Flag register 1
TCTL2   EQU     $1021   ; Timer Control register 2
TMSK1   EQU     $1022   ; main Timer interrupt Mask register 1

TOC4INT EQU     $E2     ; Timer Output Compare 4
TIC1INT EQU     $EE     ; Timer Input Capture 1

        org     MAIN_START

ir_timer                fcb     0       ; time since last bit
ir_shift                fcb     0       ; received data in progress
ir_phase                fcb     0       ; bits left to receive
ir_count                fcb     0       ; #bytes rcv'd since last check
ir_data                 fcb     0       ; full data byte as received
last_captured_input     fdb     0       ; timestamp of last bit

subroutine_initialize_module:
        ldx     #$bf00          ; pointer to interrupt vectors

* install IR detection routine
        ldd     #ir_routine
        std     TIC1INT,X

* install millisec routine
        ldd     TOC4INT,X               
        std     millisec_exit+1         ; record previous vector
        ldd     #millisec_routine
        std     TOC4INT,X               ; install new vector

        rts

* count up to 256 ms to receive single IR byte;
* otherwise clear incoming data
millisec_routine
        inc     ir_timer
        bne     millisec_exit
        dec     ir_timer        ; ir_timer is 255 now

millisec_exit
        JMP     $0000   ; this value poked in by init routine

* the ir routine is called whenever a falling edge
* (i.e., beginning of valid ir signal) is received
ir_routine
        ldaa    #4
        staa    TFLG1           ; clear tic1 interrupt flag
        
        ldaa    ir_timer
        cmpa    #5
        bmi     irgetbit        ; get bit if < 5 ms have elapsed
irreset ldaa    #9              ; capture 9 bits (8 data + 1 start)
        staa    ir_phase
        bra     ircapture

irgetbit ldaa   ir_phase        ; if ir_phase is zero, don't read bit
        beq     ircapture       ; because didn't get 5ms gap between bytes
        ldd     TIC1
        subd    last_captured_input
        cmpd    #3000
        ror     ir_shift        ; shift carry bit into answer
        dec     ir_phase
        bne     ircapture       ; continue until all bits read

irdone  ldaa    ir_shift        
        coma                    ; ones complement the answer
        staa    ir_data
        inc     ir_count
        bne     ircapture
        dec     ir_count        ; max out at 255

ircapture ldd   TIC1
        std     last_captured_input
        clr     ir_timer
        rti

* returns IR value received since last call
subroutine_sony_rcv_data:
        ldd     ir_count        ; count + data in D
        clr     ir_count
        rts

* call with 1 to turn on; 0 to turn off
subroutine_sony_rcv_init
        ldx     #$1000
        tstb
        beq     sony_off
        bset    TMSK1,X $04     * enable tic1 interrupt
        bset    TCTL2,X $20     * on falling edges (i.e., IR detects)
        bclr    TCTL2,X $10     
        clr     ir_count
        clr     ir_data
        rts
sony_off
        bclr    TMSK1,X $04
        rts
